#include <openssl/evp.h>
#include <openssl/err.h>
#include <string.h>
#include <stdio.h>

#define CHACHA_KEY_LEN 32
#define CHACHA_IV_LEN  12
#define CHACHA_TAG_LEN 16

/**
 * ENCRYPTION FUNCTION
 * Returns ciphertext length on success, -1 on failure.
 */
int chacha20_poly1305_encrypt(unsigned char *plaintext, int plaintext_len,
                              unsigned char *aad, int aad_len,
                              unsigned char *key, unsigned char *iv,
                              unsigned char *ciphertext, unsigned char *tag) {
    EVP_CIPHER_CTX *ctx;
    int len, ciphertext_len;

    if (!(ctx = EVP_CIPHER_CTX_new())) return -1;

    if (1 != EVP_EncryptInit_ex(ctx, EVP_chacha20_poly1305(), NULL, NULL, NULL)) goto err;
    if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_AEAD_SET_IVLEN, CHACHA_IV_LEN, NULL)) goto err;
    if (1 != EVP_EncryptInit_ex(ctx, NULL, NULL, key, iv)) goto err;

    // Optional AAD (e.g., packet headers)
    if (aad && aad_len > 0) {
        if (1 != EVP_EncryptUpdate(ctx, NULL, &len, aad, aad_len)) goto err;
    }

    if (1 != EVP_EncryptUpdate(ctx, ciphertext, &len, plaintext, plaintext_len)) goto err;
    ciphertext_len = len;

    if (1 != EVP_EncryptFinal_ex(ctx, ciphertext + len, &len)) goto err;
    ciphertext_len += len;

    // Extract the authentication tag
    if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_AEAD_GET_TAG, CHACHA_TAG_LEN, tag)) goto err;

    EVP_CIPHER_CTX_free(ctx);
    return ciphertext_len;

err:
    EVP_CIPHER_CTX_free(ctx);
    return -1;
}

/**
 * DECRYPTION FUNCTION
 * Returns plaintext length on success, -1 on authentication failure or error.
 */
int chacha20_poly1305_decrypt(unsigned char *ciphertext, int ciphertext_len,
                              unsigned char *aad, int aad_len,
                              unsigned char *tag, unsigned char *key, 
                              unsigned char *iv, unsigned char *plaintext) {
    EVP_CIPHER_CTX *ctx;
    int len, plaintext_len, ret;

    if (!(ctx = EVP_CIPHER_CTX_new())) return -1;

    if (1 != EVP_DecryptInit_ex(ctx, EVP_chacha20_poly1305(), NULL, NULL, NULL)) goto err;
    if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_AEAD_SET_IVLEN, CHACHA_IV_LEN, NULL)) goto err;
    if (1 != EVP_DecryptInit_ex(ctx, NULL, NULL, key, iv)) goto err;

    // Set the expected tag for verification
    if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_AEAD_SET_TAG, CHACHA_TAG_LEN, tag)) goto err;

    if (aad && aad_len > 0) {
        if (1 != EVP_DecryptUpdate(ctx, NULL, &len, aad, aad_len)) goto err;
    }

    if (1 != EVP_DecryptUpdate(ctx, plaintext, &len, ciphertext, ciphertext_len)) goto err;
    plaintext_len = len;

    // This step verifies the tag; returns 0 if data was tampered with
    ret = EVP_DecryptFinal_ex(ctx, plaintext + len, &len);

    EVP_CIPHER_CTX_free(ctx);

    if (ret > 0) {
        plaintext_len += len;
        return plaintext_len;
    } else {
        return -1; // Auth failure
    }

err:
    EVP_CIPHER_CTX_free(ctx);
    return -1;
}

int main() {
    unsigned char key[CHACHA_KEY_LEN] = "01234567890123456789012345678901";
    unsigned char iv[CHACHA_IV_LEN]   = "123456789012";
    unsigned char *data = (unsigned char *)"Modular Encryption Example";
    
    unsigned char ciphertext[128];
    unsigned char decrypted[128];
    unsigned char tag[CHACHA_TAG_LEN];

    // Encrypt
    int c_len = chacha20_poly1305_encrypt(data, strlen((char*)data), NULL, 0, key, iv, ciphertext, tag);
    
    // Decrypt
    int p_len = chacha20_poly1305_decrypt(ciphertext, c_len, NULL, 0, tag, key, iv, decrypted);

    if (p_len != -1) {
        decrypted[p_len] = '\0';
        printf("Success: %s\n", decrypted);
    } else {
        printf("Verification failed!\n");
    }
    return 0;
}
