#include <openssl/evp.h>
#include <openssl/err.h>
#include <string.h>
#include <stdio.h>

/* AES-256-XTS requires a 512-bit key (64 bytes) 
   It's actually two 256-bit keys concatenated. */
#define XTS_KEY_LEN 64
#define IV_LEN      16
#define SECTOR_SIZE 512

void handle_errors() {
    ERR_print_errors_fp(stderr);
    abort();
}

/**
 * Encrypts a block using AES-256-XTS.
 * Note: XTS is a block-to-block cipher; ciphertext size = plaintext size.
 */
int aes_xts_encrypt(unsigned char *plaintext, int plaintext_len,
                    unsigned char *key, unsigned char *iv,
                    unsigned char *ciphertext) {
    EVP_CIPHER_CTX *ctx;
    int len, ciphertext_len;

    if (!(ctx = EVP_CIPHER_CTX_new())) handle_errors();

    // Initialize encryption with AES-256-XTS
    if (1 != EVP_EncryptInit_ex(ctx, EVP_aes_256_xts(), NULL, key, iv)) handle_errors();

    // XTS does not use padding because it's designed for fixed-size sectors
    EVP_CIPHER_CTX_set_padding(ctx, 0);

    if (1 != EVP_EncryptUpdate(ctx, ciphertext, &len, plaintext, plaintext_len)) handle_errors();
    ciphertext_len = len;

    if (1 != EVP_EncryptFinal_ex(ctx, ciphertext + len, &len)) handle_errors();
    ciphertext_len += len;

    EVP_CIPHER_CTX_free(ctx);
    return ciphertext_len;
}

/**
 * Decrypts a block using AES-256-XTS.
 */
int aes_xts_decrypt(unsigned char *ciphertext, int ciphertext_len,
                    unsigned char *key, unsigned char *iv,
                    unsigned char *plaintext) {
    EVP_CIPHER_CTX *ctx;
    int len, plaintext_len;

    if (!(ctx = EVP_CIPHER_CTX_new())) handle_errors();

    if (1 != EVP_DecryptInit_ex(ctx, EVP_aes_256_xts(), NULL, key, iv)) handle_errors();
    
    EVP_CIPHER_CTX_set_padding(ctx, 0);

    if (1 != EVP_DecryptUpdate(ctx, plaintext, &len, ciphertext, ciphertext_len)) handle_errors();
    plaintext_len = len;

    if (1 != EVP_DecryptFinal_ex(ctx, plaintext + len, &len)) handle_errors();
    plaintext_len += len;

    EVP_CIPHER_CTX_free(ctx);
    return plaintext_len;
}

int main() {
    // 1. Setup Keys and Data
    unsigned char key[XTS_KEY_LEN];
    unsigned char iv[IV_LEN]; // In XTS, this is usually the Sector Number (Tweak)
    
    // Fill key and IV with dummy data for example (use RAND_bytes in production)
    memset(key, 0x01, XTS_KEY_LEN);
    memset(iv, 0x00, IV_LEN); 

    unsigned char plaintext[SECTOR_SIZE];
    unsigned char ciphertext[SECTOR_SIZE];
    unsigned char decrypted[SECTOR_SIZE];

    strcpy((char*)plaintext, "This is secret data stored on a disk sector encrypted with AES-XTS.");

    // 2. Encrypt
    int c_len = aes_xts_encrypt(plaintext, SECTOR_SIZE, key, iv, ciphertext);
    printf("Encryption complete. Ciphertext size: %d bytes\n", c_len);

    // 3. Decrypt
    int p_len = aes_xts_decrypt(ciphertext, c_len, key, iv, decrypted);
    
    if (p_len > 0) {
        printf("Decrypted text: %s\n", decrypted);
    }

    return 0;
}
