#include <openssl/evp.h>
#include <openssl/err.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define BUFSIZE 1024
#define TAG_SIZE 16
#define KEY_SIZE 32
#define IV_SIZE 12  // Рекомендований розмір IV для GCM - 12 байт

void handleErrors() {
    ERR_print_errors_fp(stderr);
    exit(1);
}

int load_binary_file(const char *path, unsigned char *out, size_t expected_len) {
    FILE *f = fopen(path, "rb");
    if (!f) {
        fprintf(stderr, "Помилка відкриття файлу: %s\n", path);
        return 0;
    }
    size_t n = fread(out, 1, expected_len, f);
    fclose(f);
    if (n != expected_len) {
        fprintf(stderr, "Помилка: файл %s має невірний розмір (%zu замість %zu)\n", path, n, expected_len);
        return 0;
    }
    return 1;
}

int save_binary_file(const char *path, unsigned char *data, size_t len) {
    FILE *f = fopen(path, "wb");
    if (!f) return 0;
    fwrite(data, 1, len, f);
    fclose(f);
    return 1;
}

int file_crypt_gcm(const char *in_p, const char *out_p, const char *tag_p,
                   unsigned char *key, unsigned char *iv, int encrypt) {
    FILE *ifp = fopen(in_p, "rb");
    FILE *ofp = fopen(out_p, "wb");
    if (!ifp || !ofp) {
        fprintf(stderr, "Помилка відкриття вхідних/вихідних файлів\n");
        return 0;
    }

    EVP_CIPHER_CTX *ctx = EVP_CIPHER_CTX_new();
    if (!ctx) handleErrors();

    // Ініціалізація GCM
    if (1 != EVP_CipherInit_ex(ctx, EVP_aes_256_gcm(), NULL, key, iv, encrypt))
        handleErrors();

    unsigned char inbuf[BUFSIZE], outbuf[BUFSIZE];
    int inlen, outlen;

    // Якщо дешифрування — спочатку завантажуємо тег
    unsigned char tag[TAG_SIZE];
    if (!encrypt) {
        if (!load_binary_file(tag_p, tag, TAG_SIZE)) return 0;
        if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_SET_TAG, TAG_SIZE, tag))
            handleErrors();
    }

    while ((inlen = fread(inbuf, 1, BUFSIZE, ifp)) > 0) {
        if (1 != EVP_CipherUpdate(ctx, outbuf, &outlen, inbuf, inlen))
            handleErrors();
        fwrite(outbuf, 1, outlen, ofp);
    }

    // Фіналізація
    if (encrypt) {
        if (1 != EVP_CipherFinal_ex(ctx, outbuf, &outlen)) handleErrors();
        // Отримуємо згенерований тег та зберігаємо його
        if (1 != EVP_CIPHER_CTX_ctrl(ctx, EVP_CTRL_GCM_GET_TAG, TAG_SIZE, tag))
            handleErrors();
        save_binary_file(tag_p, tag, TAG_SIZE);
    } else {
        // Перевірка тегу при дешифруванні
        if (EVP_CipherFinal_ex(ctx, outbuf, &outlen) <= 0) {
            fprintf(stderr, "АВТЕНТИФІКАЦІЯ ПРОВАЛЕНА: дані змінено або ключ невірний!\n");
            EVP_CIPHER_CTX_free(ctx);
            fclose(ifp); fclose(ofp);
            return 0;
        }
    }

    EVP_CIPHER_CTX_free(ctx);
    fclose(ifp); fclose(ofp);
    return 1;
                   }

                   int main(int argc, char *argv[]) {
                       if (argc < 7) {
                           fprintf(stderr, "Використання: %s <enc|dec> <in> <out> <key> <iv> <tag>\n", argv[0]);
                           return 1;
                       }

                       int encrypt = (strcmp(argv[1], "enc") == 0);
                       unsigned char key[KEY_SIZE];
                       unsigned char iv[IV_SIZE];

                       if (!load_binary_file(argv[4], key, KEY_SIZE) ||
                           !load_binary_file(argv[5], iv, IV_SIZE)) {
                           return 1;
                           }

                           if (file_crypt_gcm(argv[2], argv[3], argv[6], key, iv, encrypt)) {
                               printf("Операція %s успішна!\n", encrypt ? "шифрування" : "дешифрування");
                           } else {
                               return 1;
                           }

                           return 0;
                   }

